<?php

class Alti_ProtectUploads_Admin {

	private $plugin_name;
	private $version;
	private $messages = array();

	/**
	 * constructor
	 * @param string $plugin_name
	 * @param string $version
	 */
	public function __construct( $plugin_name, $version ) {
		$this->plugin_name = $plugin_name;
		$this->version = $version;
	}

	public function get_plugin_name() {
		return $this->plugin_name;
	}

	/**
	 * Add submenu to left page in admin
	 */
	public function add_submenu_page() {
		add_submenu_page( 'upload.php', $this->plugin_name, 'Protect Uploads <span class="dashicons dashicons-shield-alt" style="font-size:15px;"></span>', 'manage_options', $this->plugin_name . '-settings-page', array($this, 'render_settings_page') );
	}

	/**
	 * Render settings page for plugin
	 */
	public function render_settings_page() {
		require plugin_dir_path( __FILE__ ) . 'views/' . $this->plugin_name . '-admin-settings-page.php';
	}

	/**
	 * prepare enqueue styles for wordpress hook
	 */
	public function enqueue_styles() {
		wp_enqueue_style( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'assets/css/protect-uploads-admin.css', array(), $this->version, 'all' );
	}

	/**
	 * prepare enqueue scripts for wordpress hook
	 */
	public function enqueue_scripts() {
		wp_enqueue_script( $this->plugin_name, plugin_dir_url( __FILE__ ) . 'assets/js/protect-uploads-admin.js', array( 'jquery' ), $this->version, false );
	}

	/**
	 * add a settings link to plugin page.
	 * @param string $links array of links
	 */
	public function add_settings_link( $links ) {
	    $settings_link = '<a href="upload.php?page=' . $this->plugin_name . '-settings-page">' . __( 'Settings' ) . '</a>';
	    array_unshift($links, $settings_link);
	  	return $links;
	}

	/**
	 * get uploads dir
	 * @return string return path
	 */
	public function get_uploads_dir() {
		$uploads_dir = wp_upload_dir();
		return $uploads_dir['basedir'];
	}

	/**
	 * get uploads folder url
	 * @return string return full url
	 */
	public function get_uploads_url() {
		$uploads_dir = wp_upload_dir();
		return $uploads_dir['baseurl'];
	}

	public function get_uploads_subdirectories() {

		$directories = scandir( self::get_uploads_dir() );
		$subs = array( self::get_uploads_dir() );

		foreach ($directories as $directory) {

			if(is_dir(self::get_uploads_dir() . '/' . $directory) && !preg_match('/^\.*$/', $directory)) {
				$subs[] = self::get_uploads_dir() . '/' . $directory;
				$subDirectories = scandir( self::get_uploads_dir() . '/' . $directory );
				foreach ($subDirectories as $subDirectory) {
					if(is_dir(self::get_uploads_dir() . '/' . $directory . '/' . $subDirectory) && !preg_match('/^\.*$/', $subDirectory)) $subs[] = self::get_uploads_dir() . '/' . $directory . '/' . $subDirectory;
				}
			}

		}
		return $subs;

	}

	/**
	 * saving form
	 */
	public function save_form( $form ) {
		if( $form['protection'] == 'index_php' ) { $this->remove_htaccess(); $this->create_index(); }
		if( $form['protection'] == 'htaccess' )  { $this->remove_index(); $this->create_htaccess(); }
		if( $form['protection'] == 'remove' )    { $this->remove_index(); $this->remove_htaccess(); }
	}


	/**
	 *
	 */
	public function create_index() {
		// check if index php does not exists
		if(!file_exists( self::get_uploads_dir() .'/index.php' )) {

			$indexContent = "<!-- \n Generated by " . $this->get_plugin_name() . " \n http://www.alticreation.com/en/protect-uploads/ \n date:" . date('d/m/Y') . "\n--><?php // Silence is golden";
			$i = 0;
			foreach (self::get_uploads_subdirectories() as $subDirectory) {

				if( !file_put_contents( $subDirectory.'/'.'index.php', $indexContent ) ) {
					$this->messages['file'][] = array(
						'message' => __('Impossible to create or modified the index.php file in '. $subDirectory, $this->get_plugin_name()),
						'type' => 'error',
						'id' => '1'
					);
				}
				else {
					update_option( $this->get_plugin_name().'-protection', 'index_php' );
					$i++;
				}

			}

			if($i == count(self::get_uploads_subdirectories())) {
				$this->messages['file'][] = array(
						'message' => __('The index.php file has been created in main folder and subfolders (two levels max).', $this->get_plugin_name()),
						'type' => 'updated'
					);
			}

		}
		// if index php already exists
		else {
			$this->messages['file'][] = array(
				'message' => 'The index.php file already exists',
				'type' => 'error',
				'id' => '2'
			);
		}
	}

	/**
	 *
	 */
	public function create_htaccess() {
		// prepare htaccess Content
		$date             = date('Y-m-d H:i.s');
		$phpv             = phpversion();

		$htaccessContent  = "\n# BEGIN " . $this->get_plugin_name() . " Plugin\n";
		$htaccessContent  .= "\tOptions -Indexes\n";
		$htaccessContent  .= "# [date={$date}] [php={$phpv}] [plugin_name=" . $this->plugin_name . "] [version={$this->version}]\n";
		$htaccessContent  .= "# END " . $this->get_plugin_name() . " Plugin\n";

		// check if htaccess does not exists
		if(!file_exists( self::get_uploads_dir() .'/.htaccess' )) {
			// if htaccess is successfuly written
			if( !file_put_contents( self::get_uploads_dir().'/'.'.htaccess', $htaccessContent ) ) {
				$this->messages['file'][] = array(
					'message' => __('Impossible to create or modified the htaccess file.', $this->get_plugin_name()),
					'type' => 'error',
					'id' => '1'
				);
			}
			// if htaccess has not been written
			else {
				update_option( $this->get_plugin_name().'-protection', 'htaccess');
				$this->messages['file'][] = array(
					'message' => __('The htaccess file has been created.', $this->get_plugin_name()),
					'type' => 'updated'
				);
			}

		}
		// if htaccess already exists
		if(file_exists( self::get_uploads_dir() .'/.htaccess') && preg_match( '/(# BEGIN protect-uploads Plugin)(.*?)(# END protect-uploads Plugin)/is', file_get_contents( self::get_uploads_dir() .'/.htaccess' ) ) == 0) {
			// if content added to existing htaccess
			if( file_put_contents( self::get_uploads_dir() .'/.htaccess', $htaccessContent, FILE_APPEND | LOCK_EX ) ) {
				update_option( $this->get_plugin_name().'-protection', 'htaccess');
				$this->messages['file'][] = array(
					'message' => __('Existing htaccess has been updated.', $this->get_plugin_name()),
					'type' => 'updated'
				);

			}
			else {
				$this->messages['file'][] = array(
					'message' => 'The existing htaccess file couldn\'t be updated. Please check file permissions.',
					'type' => 'error',
					'id' => '2'
				);
			}
		}
	}

	/**
	 * remove index file
	 */
	public function remove_index() {

		$i = 0;
		foreach( self::get_uploads_subdirectories() as $subDirectory ) {
			if(file_exists( $subDirectory .'/index.php' )) {
				unlink( $subDirectory .'/index.php' );
				$i++;
			}

		}
		if( $i == count(self::get_uploads_subdirectories()) ) {
			$this->messages['file'][] = array(
				'message' => __('The index.php file(s) have(has) been deleted.', $this->get_plugin_name()),
				'type' => 'updated'
			);
			update_option( $this->get_plugin_name().'-protection', 'remove');
		}


	}

	/**
	 * remove htaccess
	 */
	public function remove_htaccess() {

		if(file_exists( self::get_uploads_dir() .'/.htaccess' )) {
			update_option( $this->get_plugin_name().'-protection', 'remove');

			$htaccessContent = file_get_contents(self::get_uploads_dir() .'/.htaccess');
			$htaccessContent = preg_replace( '/(# BEGIN protect-uploads Plugin)(.*?)(# END protect-uploads Plugin)/is', '', $htaccessContent );
			file_put_contents( self::get_uploads_dir() .'/.htaccess', $htaccessContent, LOCK_EX );

			// if htaccess is empty, we remove it.
			if( strlen(preg_replace("/(^[\r\n]*|[\r\n]+)[\s\t]*[\r\n]+/", "", file_get_contents(self::get_uploads_dir() .'/.htaccess'))) == 0) {
				unlink( self::get_uploads_dir() .'/.htaccess' );
			}


			//
			$this->messages['file'][] = array(
				'message' => __('The htaccess file has been updated.', $this->get_plugin_name()),
				'type' => 'updated'
			);
		}

	}

	/**
	 * check if apache is active
	 * @return message messages
	 */
	public function get_current_protection() {
		// check if header is 200 (ok)
		$uploads_headers = @get_headers( self::get_uploads_url() . '/' );
		if(!is_array($uploads_headers)) $uploads_headers[0] = '';
		if( preg_match('/200/i', $uploads_headers[0] )) {
			// because
			if( !file_exists( self::get_uploads_dir() .'/index.php' ) ) {
				return false;
			}
			else {
				return true;
			}
		}
		// check if header is 403 (forbidden)
		if( preg_match('/403/i', $uploads_headers[0] )) {
			return true;
		}

	}

	/**
	 * check if apache is active
	 * @return message messages
	 */
	public function check_apache() {

		if( !function_exists('apache_get_modules')) {
			$this->messages['apache'][] = array(
					'message' => __('The Protect Uploads plugin cannot work without Apache. Yourself or your web host has to activate this module.', $this->plugin_name),
					'type' => 'error',
					'id' => '3'
				);
		}
	}



	/**
	 * display messages manager
	 * @return array push array messages in to partial view
	 */
	public function display_messages() {

		foreach ($this->messages as $name => $messages) {
			foreach ($messages as $message) {
				require plugin_dir_path( dirname( __FILE__ ) ) . 'admin/views/includes/protect-uploads-admin-message.php';
			}
		}

	}

}